/**
  ******************************************************************************
  * @file    cp_netspec.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   network specific functions in the communication layer. This is for
  *          the BSD API as implemented for example in lwip.
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

#include "project.h"

#include "cp.h"

#define CP_USERDEFINEDLOCALTCPADDRESS  ((192<<24)+(168<<16)+(2<<8)+10)
#define CP_USERDEFINEDLOCALTCPNETMASK  ((255<<24)+(255<<16)+(0<<8)+0)
#define CP_USERDEFINEDLOCALTCPGATEWAY  ((192<<24)+(168<<16)+(2<<8)+1)


struct netif g_Netif; /* network interface structure */

unsigned long g_NetInitializers;

/** @brief Callback function to setup the network stack *
 *
 *  @return none
 *
 */

void NetInit(void)
{
    struct ip_addr ipaddr;
    struct ip_addr netmask;
    struct ip_addr gw;
    // reentrancy prevention
    if (g_NetInitializers++)
        return;
    IP4_ADDR(&ipaddr,CP_USERDEFINEDLOCALTCPADDRESS>>24,CP_USERDEFINEDLOCALTCPADDRESS>>16,CP_USERDEFINEDLOCALTCPADDRESS>>8,CP_USERDEFINEDLOCALTCPADDRESS&0xff);
    IP4_ADDR(&netmask,CP_USERDEFINEDLOCALTCPNETMASK>>24,CP_USERDEFINEDLOCALTCPNETMASK>>16,CP_USERDEFINEDLOCALTCPNETMASK>>8,CP_USERDEFINEDLOCALTCPNETMASK&0xff);
    IP4_ADDR(&gw,CP_USERDEFINEDLOCALTCPGATEWAY>>24,CP_USERDEFINEDLOCALTCPGATEWAY>>16,CP_USERDEFINEDLOCALTCPGATEWAY>>8,CP_USERDEFINEDLOCALTCPGATEWAY&0xff);    
    tcpip_init( NULL, NULL );	
    netif_add(&g_Netif, &ipaddr, &netmask, &gw, NULL, &ethernetif_init, &tcpip_input);
    netif_set_default(&g_Netif);
    while (!(g_Netif.flags & NETIF_FLAG_LINK_UP))
        vTaskDelay(2);
    netif_set_up(&g_Netif);
}

/** @brief outputs a single character over a network connection.
 *
 *  @param p_Char character to pass to output stream
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 */

CP_STATUSCODE CP_EmitCharVerbatim(unsigned char p_Char,PCP_PROCESSOR p_Processor)
{
    volatile int a_LenSent = send(p_Processor->m_Socket,&p_Char,1,0);
    if (a_LenSent == 1)
        return CP_STATUSCODE_SUCCESS;      
    return CP_STATUSCODE_CONNLOSS;
}

/** @brief outputs a packet of characters over a network connection.
 *
 *  @param p_Chunk characters to pass to output stream
 *  @param p_ChunkLen Length of chunk
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * to be really robust, we must also implement a timeout here!
 */

CP_STATUSCODE CP_EmitChunkVerbatim(unsigned char *p_Chunk,unsigned long p_ChunkLen,PCP_PROCESSOR p_Processor)
{
    unsigned long a_WrittenSoFar = 0;
    while (a_WrittenSoFar < p_ChunkLen)
    {
        int a_LenSent = send(p_Processor->m_Socket,&p_Chunk[a_WrittenSoFar],p_ChunkLen-a_WrittenSoFar,0); 
        if (a_LenSent < 0)
            return CP_STATUSCODE_CONNLOSS;
        a_WrittenSoFar += a_LenSent;    
    }      
    return CP_STATUSCODE_SUCCESS;
}
